
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.lang.Math;

public class TileMap
{

    // represents different tiles in the data array

	private static final int INTERSECT_WIRE_CROSS = 13;
	private static final int INTERSECT_WIRE_N = 12;
	private static final int INTERSECT_WIRE_W = 11;
	private static final int INTERSECT_WIRE_S = 10;
	private static final int INTERSECT_WIRE_E = 9;
	private static final int CORNER_WIRE_LL = 8;
	private static final int CORNER_WIRE_LR = 7;
	private static final int CORNER_WIRE_UR = 6;
	private static final int CORNER_WIRE_UL = 5;
	private static final int HORIZONTAL_WIRE = 4;
	private static final int VERTICAL_WIRE = 3;
	private static final int POWERCELL_BOTTOM = 2;
	private static final int POWERCELL_TOP = 1;

    // placeholder for a cell with no tile

	private static final int NO_TILE = 0;

	// number of tiles in map in horizontal direction

	private static final int WIDTH = 13;
    // number of tiles in map in vertical direction

	private static final int HEIGHT = 11;

    // Original tile size is 55. Attempting to scale to 50 for player speed consistency
	public static final double TILE_SIZE = 50;

	// map data
	private int[][] data = {{5,4,4,4,4,4,10,4,4,4,4,4,6},
                            {3,0,0,0,0,0,3,0,0,0,0,0,3},
                            {3,0,0,0,0,0,9,4,4,4,4,4,11},
                            {3,0,0,0,0,0,3,0,0,0,0,0,3},
                            {3,0,0,0,0,0,9,4,4,4,4,4,11},
                            {1,0,0,0,0,0,3,0,0,0,0,0,3},
                            {2,0,0,0,0,0,9,4,4,4,4,4,11},
                            {3,0,0,0,0,0,3,0,0,0,0,0,3},
                            {3,0,0,0,0,0,9,4,4,4,4,4,11},
                            {3,0,0,0,0,0,3,0,0,0,0,0,3},
                            {8,4,4,4,4,4,12,4,4,4,4,4,7}};

	private MediaManager mediaManager;
	private BufferedImage tile;
	private CircuitConductor cc;

	public TileMap(CircuitConductor cc)
	{
		mediaManager = new MediaManager();
		this.cc = cc;
	}

	// draw the map based on the tile data
	public void paint(Graphics2D g)
	{

		for (int x=0;x<HEIGHT;x++)
		{
			for (int y=0;y<WIDTH;y++)
			{

                // in this scheme, data array contents are taken going vertically
                // down each column and their corresponding tiles are painted across
                // the screen horizontally. Effectively, the data's position in the
                // array is "mirrored" in comparison to where its corresponding
                // tile is painted visually.

                int tileType = data[x][y];
				switch (tileType)
				{
					case NO_TILE:
					    tile = null;
					    break;
					case POWERCELL_TOP:
					    tile = mediaManager.getBitmap("powerCellHiGlowTile.gif");
					    break;
					case POWERCELL_BOTTOM:
						tile = mediaManager.getBitmap("powerCellLowGlowTile.gif");
					    break;
					case VERTICAL_WIRE:
						tile = mediaManager.getBitmap("vertWireGlowTile.gif");
					    break;
					case HORIZONTAL_WIRE:
						tile = mediaManager.getBitmap("horiWireGlowTile.gif");
					    break;
					case CORNER_WIRE_UL:
						tile = mediaManager.getBitmap("cornerWireGlowTile1.gif");
					    break;
					case CORNER_WIRE_UR:
				        tile = mediaManager.getBitmap("cornerWireGlowTile2.gif");
					    break;
					case CORNER_WIRE_LR:
						tile = mediaManager.getBitmap("cornerWireGlowTile3.gif");
					    break;
					case CORNER_WIRE_LL:
						tile = mediaManager.getBitmap("cornerWireGlowTile4.gif");
					    break;
					case INTERSECT_WIRE_E:
						tile = mediaManager.getBitmap("intersectWireGlowTile1.gif");
					    break;
					case INTERSECT_WIRE_S:
						tile = mediaManager.getBitmap("intersectWireGlowTile2.gif");
					    break;
				    case INTERSECT_WIRE_W:
						tile = mediaManager.getBitmap("intersectWireGlowTile3.gif");
					    break;
					case INTERSECT_WIRE_N:
						tile = mediaManager.getBitmap("intersectWireGlowTile4.gif");
					    break;
					case INTERSECT_WIRE_CROSS:
						tile = mediaManager.getBitmap("crossIntersectWireTile.gif");
					    break;
                }

                if(tile != null)
                {
					g.drawImage(tile,y*(int)TILE_SIZE,65 + x*(int)TILE_SIZE,
					            y*(int)TILE_SIZE + 50,65 + x*(int)TILE_SIZE + 50,
					            1,1,54,54,cc);
			    }
			}
		}
	}

    // returns whether or not the particular area of the tile to move to
    // is blocked for movement on the x-axis
	public boolean xBlocked(double xPos, double yPos, double vx, double vy,
	                        double objectWidth, double objectHeight,
	                        GameObject gameobj){


        // get coordinates of center of player
        // These adjustments will need to be different for each gameObject

        System.out.println("xPos = " + xPos + ", yPos = " + yPos);

//        System.out.println("objectWidth = " + objectWidth + ", objectHeight = " +objectHeight);

        double centeredX = getCenterXCoordinate(xPos, objectWidth);
        double centeredY = getCenterYCoordinate(yPos, objectHeight);

//        System.out.println("centeredX = " + centeredX + ", centeredY = " +centeredY);

        // get coordinates of player with respect to position inside tile

        double xInTile = getInTileXCoordinate(centeredX);
		double yInTile = getInTileYCoordinate(centeredY);

		System.out.println("xInTile = " +xInTile+ ", yInTile = " +yInTile);

		// get back tile's data coordinates. NOTE: this gets back the position of
		// the relevant data in the array, the x and y coordinates of which are
		// "mirrored" (essentially swapped) compared to the corresponding painted
		// tile onscreen.

		double xCoordinate = (centeredY - 65) / TILE_SIZE;
		double yCoordinate = centeredX / TILE_SIZE;

		int tileType = data[(int)xCoordinate][(int)yCoordinate];

//		System.out.println("data[" +(int)xCoordinate + "]["
//		                   +(int)yCoordinate +"] = " + tileType);

//        System.out.println("vx = " +vx+ ", vy = " +vy);


		switch (tileType)
		{
			case POWERCELL_TOP:
			case POWERCELL_BOTTOM:
			case VERTICAL_WIRE:
			   if(vx != 0)    // if attempting movement to left or right
			      return true;
			   break;
		    case HORIZONTAL_WIRE:
               // No case needed for horizontal wire
		       break;
		    case CORNER_WIRE_UL:
		       if(((vx != 0) && ((xInTile + vx) < TILE_SIZE/2)) ||
                  ((vx != 0) && (yInTile > TILE_SIZE/2)))
		          return true;
		       break;
		    case CORNER_WIRE_UR:
		       if(((vx != 0) && ((xInTile + vx) > TILE_SIZE/2)) ||
		          ((vx != 0) && (yInTile > TILE_SIZE/2)))
		          return true;
		       break;
		    case CORNER_WIRE_LR:
		       if(((vx != 0) && ((xInTile + vx) > TILE_SIZE/2)) ||
		          ((vx != 0) && (yInTile < TILE_SIZE/2)))
		          return true;
		       break;
		    case CORNER_WIRE_LL:
		       if(((vx != 0) && ((xInTile + vx) < TILE_SIZE/2)) ||
		          ((vx != 0) && (yInTile < TILE_SIZE/2)))
		          return true;
		       break;
		    case INTERSECT_WIRE_E:
		       if(((vx != 0) && ((xInTile + vx) < TILE_SIZE/2)) ||
		          ((vx != 0) && (yInTile != TILE_SIZE/2)))
		          return true;
		       break;
		    case INTERSECT_WIRE_S:
		    case INTERSECT_WIRE_N:
		       if((vx != 0) && (yInTile != TILE_SIZE/2))
		          return true;
		       break;
		    case INTERSECT_WIRE_W:
		       if(((vx != 0) && ((xInTile + vx) > TILE_SIZE/2)) ||
		          ((vx != 0) && (yInTile != TILE_SIZE/2)))
		          return true;
		       break;
		    case INTERSECT_WIRE_CROSS:
		       if((vx != 0) && (yInTile != TILE_SIZE/2))
		          return true;
		       break;

	//	    default:
	//	       return false;

		}

		return false;
	}

    // returns whether or not the particular area of the tile to move to
    // is blocked for movement on the y-axis
	public boolean yBlocked(double xPos, double yPos, double vx, double vy, double objectWidth,
		                       double objectHeight){

	        // get coordinates of center of player
	        // These adjustments will need to be different for each gameObject

//	        System.out.println("objectWidth = " + objectWidth + ", objectHeight = " +objectHeight);

            double centeredX = getCenterXCoordinate(xPos, objectWidth);
            double centeredY = getCenterYCoordinate(yPos, objectHeight);

//	        System.out.println("centeredX = " + centeredX + ", centeredY = " +centeredY);

	        // get coordinates of player with respect to position inside tile

            double xInTile = getInTileXCoordinate(centeredX);
		    double yInTile = getInTileYCoordinate(centeredY);

//			System.out.println("xInTile = " +xInTile+ ", yInTile = " +yInTile);

		    // get back tile's data coordinates. NOTE: this gets back the position of
		    // the relevant data in the array, the x and y coordinates of which are
		    // "mirrored" (essentially swapped) compared to the corresonding painted
		    // tile onscreen.

			double xCoordinate = (centeredY - 65) / TILE_SIZE;
			double yCoordinate = centeredX / TILE_SIZE;

			int tileType = data[(int)xCoordinate][(int)yCoordinate];

//			System.out.println("data[" +(int)xCoordinate + "]["
//			                   +(int)yCoordinate +"] = " + tileType);

//	        System.out.println("vx = " +vx+ ", vy = " +vy);

			switch (tileType)
			{
				case POWERCELL_TOP:
				   // May need case to prevent movement past end
				  if(yInTile + vy > 35)
				      return true;
				      break;
				case POWERCELL_BOTTOM:
				  if (yInTile + vy < 15.0)
				      return true;
				      break;
				case VERTICAL_WIRE:
				   // No case needed for vertical wire
			          break;
		        case HORIZONTAL_WIRE:
		           if(vy != 0)    // if attempting movement up or down
		              return true;
		              break;
		        case CORNER_WIRE_UL:
		           if(((vy != 0) && ((yInTile + vy) < TILE_SIZE/2)) ||
					  ((vy != 0) && (xInTile > TILE_SIZE/2)))
		              return true;
		              break;
		        case CORNER_WIRE_UR:
		           if(((vy != 0) && ((yInTile + vy) < TILE_SIZE/2)) ||
		              ((vy != 0) && (xInTile < TILE_SIZE/2)))
		              return true;
		              break;
		        case CORNER_WIRE_LR:
		           if(((vy != 0) && ((yInTile + vy) > TILE_SIZE/2)) ||
		              ((vy != 0) && (xInTile < TILE_SIZE/2)))
		              return true;
		              break;
		        case CORNER_WIRE_LL:
		           if(((vy != 0) && ((yInTile + vy) > TILE_SIZE/2)) ||
		              ((vy != 0) && (xInTile > TILE_SIZE/2)))
		              return true;
		              break;
		        case INTERSECT_WIRE_E:
		        case INTERSECT_WIRE_W:
		          if((vy != 0) && (xInTile != TILE_SIZE/2))
		              return true;
		              break;
		        case INTERSECT_WIRE_S:
		          if(((vy != 0) && ((yInTile + vy) < TILE_SIZE/2)) ||
		             ((vy != 0) && (xInTile != TILE_SIZE/2)))
		              return true;
		              break;
		        case INTERSECT_WIRE_N:
		          if(((vy != 0) && ((yInTile + vy) > TILE_SIZE/2)) ||
		             ((vy != 0) && (xInTile != TILE_SIZE/2)))
		              return true;
		              break;
		        case INTERSECT_WIRE_CROSS:
		          if((vy != 0) && (xInTile != TILE_SIZE/2))
		              return true;
		              break;
			}
			return false;
		}

		public double getCenterXCoordinate(double xOriginCoordinate,
		                                   double objectWidth)
		{
			double centerXCoordinate = xOriginCoordinate + (objectWidth/2);
			return centerXCoordinate;
		}

		public double getCenterYCoordinate(double yOriginCoordinate,
		                                   double objectHeight)
		{
			double centerYCoordinate = yOriginCoordinate + (objectHeight/2);
			return centerYCoordinate;
		}

        // coordinates with respect to tile need the coordinates of the center
        // of the gameObject as inputs

		public double getInTileXCoordinate(double centeredXIn)
        {
	        double xWithRespectToTile = centeredXIn % TILE_SIZE;
	        return xWithRespectToTile;
	    }

		public double getInTileYCoordinate(double centeredXIn)
        {
	        double yWithRespectToTile = (centeredXIn - 65) % TILE_SIZE;
	        return yWithRespectToTile;
	    }

	    public double getTileSize()
	    {
			return TILE_SIZE;
		}

}
